<?php
require_once 'php/auth_check.php'; // Ensure user is logged in
require_once 'php/db_connect.php';

// Check if a mall is selected
if (!isset($_GET['mall']) || empty($_GET['mall'])) {
    // No mall selected, redirect to mall selection page
    header("Location: mall_selection.php");
    exit;
}

$selected_mall_slug = $_GET['mall'];
$mall_name = "";
$mall_id = null;

// Fetch mall details based on slug
$sql_mall = "SELECT id, name FROM malls WHERE slug = ?";
if ($stmt_mall = $conn->prepare($sql_mall)) {
    $stmt_mall->bind_param("s", $selected_mall_slug);
    if ($stmt_mall->execute()) {
        $result_mall = $stmt_mall->get_result();
        if ($result_mall->num_rows == 1) {
            $mall_row = $result_mall->fetch_assoc();
            $mall_name = $mall_row['name'];
            $mall_id = $mall_row['id'];
        } else {
            // Mall not found, redirect or show error
            $_SESSION['error_message'] = "Invalid mall selected.";
            header("Location: mall_selection.php");
            exit;
        }
    } else {
        // SQL error
        $_SESSION['error_message'] = "Error fetching mall details.";
        header("Location: mall_selection.php");
        exit;
    }
    $stmt_mall->close();
} else {
    $_SESSION['error_message'] = "Database error preparing mall query.";
    header("Location: mall_selection.php");
    exit;
}

// Store selected mall in session for JavaScript to potentially access or for other pages
$_SESSION['selected_mall_id'] = $mall_id;
$_SESSION['selected_mall_name'] = $mall_name;
$_SESSION['selected_mall_slug'] = $selected_mall_slug;

$pageTitle = htmlspecialchars($mall_name) . " - Leasing Information";

// Fetch distinct levels for the selected mall that have units
$levels = [];
$sql_levels = "SELECT DISTINCT level FROM units WHERE mall_id = ? ORDER BY CAST(REPLACE(level, 'L', '') AS UNSIGNED)";
if($stmt_levels = $conn->prepare($sql_levels)){
    $stmt_levels->bind_param("i", $mall_id);
    if($stmt_levels->execute()){
        $result_levels = $stmt_levels->get_result();
        while($row_level = $result_levels->fetch_assoc()){
            $levels[] = $row_level['level'];
        }
    }
    $stmt_levels->close();
}

// Default to first level if available, or L1
$default_level = !empty($levels) ? $levels[0] : '1'; // Note: '1' not 'L1' to match data-level
                                                     // Or adjust your data-level to 'L1', 'L2' etc.
                                                     // For consistency, let's assume data-level is just the number.
                                                     // And unit levels are stored as '1', '2', '3'
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0"> <!-- Added for responsiveness -->
    <title><?php echo $pageTitle; ?></title>
    <link rel="stylesheet" href="css/style.css">
    <!-- Using Google Fonts for a more modern and professional look -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <!-- Font Awesome for icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        /* CSS Variables for easy theme management, consistent with other pages */
        :root {
            --primary-color: #007bff; /* A standard blue for primary actions - now used for header text/icons */
            --primary-hover-color: #0056b3; /* Darker blue on hover */
            --background-color: #f8f9fa; /* Light grey background */
            --card-background: #ffffff; /* White card background */
            --text-color: #343a40; /* Dark grey for general text */
            --heading-color: #212529; /* Even darker for headings - now used for header/footer background */
            --border-color: #e9ecef; /* Light grey border */
            --shadow-color: rgba(0, 0, 0, 0.08); /* Subtle shadow */
            --link-color: #007bff; /* Blue for links */
            --link-hover-color: #0056b3; /* Darker blue for link hover */
            --accent-color: #28a745; /* Green for success/accent */
            --accent-hover-color: #218838;
        }

        body {
            font-family: 'Inter', sans-serif;
            margin: 0;
            padding: 0;
            display: flex;
            flex-direction: column;
            min-height: 100vh;
            background-color: var(--background-color);
            color: var(--text-color);
            line-height: 1.6;
            box-sizing: border-box;
        }

        header {
            background-color: var(--heading-color); /* Changed to match footer */
            color: white;
            padding: 1.5rem 2rem;
            text-align: center;
            box-shadow: 0 4px 8px var(--shadow-color);
            width: 100%;
            box-sizing: border-box;
            position: sticky;
            top: 0;
            z-index: 1000;
            display: flex;
            flex-direction: column; /* Keep column for title first */
            align-items: center;
        }

        header h1 {
            margin: 0 0 1.5rem 0; /* More space below title */
            font-size: 2em;
            font-weight: 700;
            color: white;
        }

        /* Container for User Menu and Level Selector to be in the same row */
        .header-controls {
            display: flex;
            justify-content: space-between; /* Space out items */
            align-items: center;
            flex-wrap: wrap; /* Allow wrapping on smaller screens */
            width: 100%; /* Take full width of header */
            max-width: 1000px; /* Limit max width for desktop */
            margin-top: 1rem; /* Space from title */
            gap: 15px; /* Space between items if they wrap */
        }

		.header-action-button {
            background-color: var(--accent-color); /* Use accent color to stand out */
            color: white;
            padding: 0.7rem 1.2rem;
            border-radius: 6px;
            text-decoration: none; /* Remove underline from link */
            font-size: 0.95em;
            font-weight: 500;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            height: fit-content; /* Ensures it aligns well with other buttons */
        }

        .header-action-button:hover {
            background-color: var(--accent-hover-color);
            transform: translateY(-2px); /* Slight lift on hover */
        }
		
        /* User Menu Styles */
        .user-menu-wrapper {
            position: relative;
            margin: 0; /* Remove previous margins */
            width: fit-content; /* Adjust width to content */
        }

        .user-menu-toggle {
            background-color: rgba(255, 255, 255, 0.2);
            color: white;
            border: none;
            padding: 0.7rem 1.2rem;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.95em;
            font-weight: 500;
            transition: background-color 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            outline: none; /* Remove outline on focus */
        }

        .user-menu-toggle:hover {
            background-color: rgba(255, 255, 255, 0.3);
        }

        .user-menu-dropdown {
            display: none; /* Hidden by default */
            position: absolute;
            top: calc(100% + 10px); /* Position below the toggle button */
            right: 0; /* Align to the right */
            background-color: var(--card-background);
            box-shadow: 0 8px 16px rgba(0,0,0,0.2);
            border-radius: 8px;
            overflow: hidden;
            min-width: 180px; /* Minimum width for dropdown */
            z-index: 1001;
            padding: 10px 0;
            border: 1px solid var(--border-color);
            text-align: left;
        }

        .user-menu-dropdown.show {
            display: block; /* Show when 'show' class is present */
        }

        .user-menu-dropdown a, .user-menu-dropdown span {
            color: var(--text-color);
            padding: 12px 20px;
            text-decoration: none;
            display: block;
            white-space: nowrap; /* Prevent wrapping */
            transition: background-color 0.2s ease;
            font-size: 0.9em;
            font-weight: 400;
        }

        .user-menu-dropdown a:hover {
            background-color: var(--background-color);
            color: var(--primary-color);
        }

        .user-menu-dropdown span {
            cursor: default; /* Not clickable */
            font-weight: 500;
            border-bottom: 1px solid var(--border-color);
            margin-bottom: 5px;
            padding-bottom: 10px;
        }

        /* Level Selector Navigation */
        #level-selector {
            display: flex;
            justify-content: center; /* Center buttons within its container */
            flex-wrap: wrap;
            gap: 10px;
            margin: 0; /* Remove previous margins */
            flex-grow: 1; /* Allow it to take available space */
        }

        #level-selector button {
            background-color: white;
            color: var(--primary-color);
            border: 1px solid var(--primary-color);
            padding: 0.7rem 1.2rem;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.95em;
            font-weight: 500;
            transition: background-color 0.3s ease, color 0.3s ease, box-shadow 0.3s ease;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            outline: none;
        }

        #level-selector button:hover {
            background-color: var(--primary-color);
            color: white;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }

        #level-selector button.active {
            background-color: var(--accent-color); /* Active level button color */
            color: white;
            border-color: var(--accent-color);
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
            font-weight: 600;
        }

        main {
            display: flex;
            flex-grow: 1;
            padding: 2rem;
            gap: 2rem;
            flex-wrap: wrap; /* Allow wrapping for smaller screens */
            justify-content: center; /* Center items when wrapped */
            max-width: 1200px; /* Max width for content */
            margin: 2rem auto; /* Center main content */
        }

        #floor-plan-area {
            flex: 2; /* Takes more space */
            min-width: 300px; /* Minimum width for floor plan */
            background-color: var(--card-background);
            border-radius: 12px;
            box-shadow: 0 8px 20px var(--shadow-color);
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden; /* Ensure content stays within bounds */
            padding: 1.5rem; /* Padding inside the card */
            box-sizing: border-box;
            border: 1px solid var(--border-color);
            /* Placeholder for floor plan image */
            min-height: 400px; /* Give it some height initially */
            color: #ccc;
            font-size: 1.2em;
        }

        #floor-plan-area img.floor-plan-thumb {
            max-width: 100%;
            height: auto;
            border-radius: 8px;
            cursor: pointer; /* Indicate it's clickable */
            transition: transform 0.2s ease;
        }

        #floor-plan-area img.floor-plan-thumb:hover {
            transform: scale(1.02); /* Slight zoom on hover */
        }


        #unit-button-container {
            flex: 1; /* Takes less space */
            min-width: 250px; /* Minimum width for unit buttons */
            max-height: 600px; /* Limit height for scrollability */
            overflow-y: auto; /* Enable scrolling if too many buttons */
            background-color: var(--card-background);
            border-radius: 12px;
            box-shadow: 0 8px 20px var(--shadow-color);
            padding: 1.5rem;
            box-sizing: border-box;
            border: 1px solid var(--border-color);
            display: flex;
            flex-direction: column;
            gap: 10px;
        }

        .unit-button {
            display: block;
            width: 100%;
            padding: 0.8rem 1rem;
            background-color: #f0f0f0;
            color: var(--text-color);
            border: 1px solid #ddd;
            border-radius: 6px;
            text-align: left;
            text-decoration: none;
            font-size: 0.95em;
            transition: background-color 0.3s ease, transform 0.2s ease;
            outline: none;
            cursor: pointer;
        }

        .unit-button:hover {
            background-color: #e0e0e0;
            transform: translateX(5px);
        }

        .unit-button.active {
            background-color: var(--primary-color);
            color: white;
            border-color: var(--primary-color);
            font-weight: 500;
        }

        #unit-details-panel {
            flex: 1; /* Equal flex with unit buttons */
            min-width: 280px; /* Minimum width for details panel */
            background-color: var(--card-background);
            border-radius: 12px;
            box-shadow: 0 8px 20px var(--shadow-color);
            padding: 1.5rem;
            box-sizing: border-box;
            border: 1px solid var(--border-color);
            display: flex; /* Use flex for content arrangement */
            flex-direction: column;
        }

        #unit-details-panel h2 {
            color: var(--heading-color);
            margin-top: 0;
            margin-bottom: 1.8rem; /* Increased margin */
            font-size: 1.6em; /* Slightly larger */
            font-weight: 700; /* Bolder */
            text-align: center;
        }

        #details-content {
            flex-grow: 1; /* Allow content to take available space */
            display: flex;
            flex-direction: column;
            justify-content: space-between; /* Push download button to bottom */
        }

        #details-content p {
            margin-bottom: 0.8rem; /* Adjusted margin */
            font-size: 1em; /* Slightly larger text */
            color: #555;
            display: flex; /* Use flex for key-value pair alignment */
            justify-content: space-between; /* Space out key and value */
            padding: 0.4rem 0; /* Add some padding for each detail line */
            border-bottom: 1px dashed var(--border-color); /* Subtle separator */
        }

        #details-content p:last-of-type {
            border-bottom: none; /* No border for the last item */
        }

        #details-content p strong {
            color: var(--text-color); /* Stronger color for labels */
            font-weight: 600;
            flex-shrink: 0; /* Prevent label from shrinking */
            margin-right: 10px; /* Space between label and value */
        }

        #details-content p span {
            text-align: right; /* Align value to the right */
            flex-grow: 1; /* Allow value to take remaining space */
            word-wrap: break-word; /* Ensure long values wrap */
        }

        #details-content .initial-message {
            text-align: center;
            padding: 2rem;
            font-style: italic;
            color: #888;
        }


        #download-button-container {
            margin-top: 2rem; /* More space from details */
            text-align: center;
            padding-top: 1.5rem; /* Padding above the button */
            border-top: 1px solid var(--border-color); /* Separator above the button */
        }

        #download-button-container button {
            background-color: var(--accent-color);
            color: white;
            padding: 0.9rem 1.8rem; /* Larger button */
            border: none;
            border-radius: 8px; /* More rounded button */
            cursor: pointer;
            font-size: 1.05em; /* Larger font */
            font-weight: 600; /* Bolder font */
            transition: background-color 0.3s ease, transform 0.2s ease, box-shadow 0.3s ease;
            box-shadow: 0 4px 10px rgba(0,0,0,0.1);
            outline: none;
        }

        #download-button-container button:hover {
            background-color: var(--accent-hover-color);
            transform: translateY(-3px);
            box-shadow: 0 6px 15px rgba(0,0,0,0.15);
        }

        footer {
            background-color: var(--heading-color);
            color: white;
            text-align: center;
            padding: 1rem 2rem;
            margin-top: auto; /* Pushes footer to the bottom */
            box-shadow: 0 -4px 8px var(--shadow-color);
            width: 100%;
            box-sizing: border-box;
            font-size: 0.85em;
        }

        /* Magnify Modal Styles - Professional and centered */
        .modal {
            display: none; /* Hidden by default */
            position: fixed; /* Stay in place */
            z-index: 2000; /* Sit on top */
            left: 0;
            top: 0;
            width: 100%; /* Full width */
            height: 100%; /* Full height */
            overflow: auto; /* Enable scroll if needed */
            background-color: rgba(0,0,0,0.7); /* Black w/ opacity */
            /* Removed display: flex here */
            align-items: center; /* Center vertically */
            justify-content: center; /* Center horizontally */
        }

        .modal.active { /* Show modal when active class is present */
            display: flex;
        }

        .modal-content {
            margin: auto;
            display: block;
            width: 90%;
            max-width: 900px; /* Max width for the magnified image */
            border-radius: 8px;
            box-shadow: 0 0 20px rgba(0,0,0,0.5);

        }

        @keyframes zoom {
            from {transform: scale(0.1)}
            to {transform: scale(1)}
        }

        .modal-close {
            position: absolute;
            top: 20px;
            right: 35px;
            color: #f1f1f1;
            font-size: 40px;
            font-weight: bold;
            transition: 0.3s;
            cursor: pointer;
        }

        .modal-close:hover,
        .modal-close:focus {
            color: #bbb;
            text-decoration: none;
        }


        /* Responsive adjustments */
        @media (max-width: 1024px) {
            .header-controls {
                flex-direction: column; /* Stack user menu and levels on medium screens */
                align-items: center;
            }
            .user-menu-wrapper {
                margin-bottom: 15px; /* Space between stacked elements */
            }

            main {
                padding: 1.5rem;
                gap: 1.5rem;
            }

            #floor-plan-area,
            #unit-button-container,
            #unit-details-panel {
                flex: 1 1 100%; /* Stack columns on medium screens */
                min-width: unset; /* Remove min-width to allow full flexibility */
                max-width: 100%;
            }

            #unit-button-container {
                max-height: 400px; /* Adjust height for stacking */
            }
        }

        @media (max-width: 768px) {
            header {
                padding: 1.2rem 1rem;
            }

            header h1 {
                font-size: 1.8em;
            }

            /* Responsive user menu toggle */
            .user-menu-toggle {
                font-size: 0.85em;
                padding: 0.6rem 1rem;
            }
            .user-menu-dropdown {
                min-width: 150px;
            }
            .user-menu-dropdown a, .user-menu-dropdown span {
                padding: 10px 15px;
                font-size: 0.85em;
            }

            #level-selector button {
                padding: 0.6rem 1rem;
                font-size: 0.9em;
            }

            main {
                padding: 1rem;
                gap: 1rem;
            }

            #floor-plan-area,
            #unit-button-container,
            #unit-details-panel {
                padding: 1rem;
            }

            #unit-details-panel h2 {
                font-size: 1.3em;
                margin-bottom: 1rem;
            }

            .unit-button {
                padding: 0.7rem 0.8rem;
                font-size: 0.9em;
            }

            #download-button-container button {
                padding: 0.7rem 1.2rem;
                font-size: 0.9em;
            }

            #details-content p {
                font-size: 0.9em;
                padding: 0.3rem 0;
            }
            #details-content p strong {
                font-size: 0.95em;
            }
            #download-button-container {
                margin-top: 1.5rem;
                padding-top: 1rem;
            }
        }

        @media (max-width: 480px) {
            header {
                padding: 1rem 0.5rem;
            }

            header h1 {
                font-size: 1.5em;
                margin-bottom: 1rem; /* Adjust space for smaller header */
            }

            /* Further responsive user menu toggle */
            .user-menu-toggle {
                font-size: 0.8em;
                padding: 0.5rem 0.8rem;
            }
            .user-menu-dropdown {
                min-width: 120px;
                padding: 5px 0;
            }
            .user-menu-dropdown a, .user-menu-dropdown span {
                padding: 8px 12px;
                font-size: 0.8em;
            }

            .header-controls {
                gap: 10px; /* Adjust gap for very small screens */
            }

            #level-selector {
                gap: 5px;
            }

            #level-selector button {
                padding: 0.5rem 0.8rem;
                font-size: 0.85em;
            }

            main {
                padding: 0.8rem;
                gap: 0.8rem;
            }

            #floor-plan-area {
                min-height: 300px;
            }

            #unit-details-panel h2 {
                font-size: 1.2em;
            }
            #details-content p {
                flex-direction: column; /* Stack key-value pairs on very small screens */
                align-items: flex-start;
                padding: 0.2rem 0;
            }
            #details-content p strong {
                margin-right: 0;
                margin-bottom: 5px; /* Space between stacked key and value */
            }
            #details-content p span {
                text-align: left;
            }
            #download-button-container {
                margin-top: 1rem;
                padding-top: 0.8rem;
            }
            #download-button-container button {
                padding: 0.7rem 1.2rem;
                font-size: 0.95em;
            }
        }
    </style>
    <script>
        const SELECTED_MALL_ID = <?php echo json_encode($mall_id); ?>;
        const SELECTED_MALL_SLUG = <?php echo json_encode($selected_mall_slug); ?>;
        const AVAILABLE_LEVELS = <?php echo json_encode($levels); ?>; // e.g., ['1', '2', '3']
    </script>
</head>
<body>

    <header>
        <h1><?php echo htmlspecialchars($mall_name); ?> - Units for Lease</h1>
        <div class="header-controls">
            <div class="user-menu-wrapper">
                <button class="user-menu-toggle" id="userMenuToggle">
                    <i class="fas fa-user-circle"></i>
                    <span>Hello, <?php echo htmlspecialchars($_SESSION["username"]); ?></span>
                    <i class="fas fa-caret-down"></i>
                </button>
                <div class="user-menu-dropdown" id="userMenuDropdown">
                    <span>Logged in as: <?php echo htmlspecialchars($_SESSION["username"]); ?></span>
                    <a href="mall_selection.php"><i class="fas fa-building"></i> Change Mall</a>
                    <?php if (isset($_SESSION["is_admin"]) && $_SESSION["is_admin"] === true): ?>
                        <a href="admin.php"><i class="fas fa-user-shield"></i> Admin Panel</a>
                    <?php endif; ?>
                    <a href="php/logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
                </div>
            </div>
            <nav id="level-selector">
                <!-- Level buttons will be dynamically inserted here by JavaScript -->
            </nav>
			    <a href="sales_kit.php" class="header-action-button" title="Download Sales Kits">
                <i class="fas fa-book-open"></i>
                <span>Sales Kit</span>
            </a>
        </div>
        </div>
    </header>

    <main>
        <div id="floor-plan-area">
            <!-- Floor plan image will be loaded here by JavaScript -->
            <p class="initial-message">Select a level to view the floor plan.</p>
        </div>

        <div id="unit-button-container">
            <!-- Unit buttons will be dynamically inserted here by JavaScript -->
            <p class="initial-message">Select a level to load units.</p>
        </div>

        <aside id="unit-details-panel">
            <h2>Unit Details</h2>
            <div id="details-content">
              <p class="initial-message">Select a mall and level, then a unit to see its details.</p>
              <div id="download-button-container">
                  <!-- Download button will be dynamically inserted here by JavaScript -->
              </div>
            </div>
        </aside>
    </main>

    <footer>
        <p>&copy; <span id="current-year"><?php echo date("Y"); ?></span> EW Private Limited. All rights reserved.</p>
    </footer>

    <div id="magnify-modal" class="modal">
        <span class="modal-close">×</span>
        <img class="modal-content" id="magnified-img" src="" alt="Magnified Floor Plan">
    </div>

    <script src="js/script.js"></script>
    <script>
        // Set the current year in the footer
        document.getElementById('current-year').textContent = new Date().getFullYear();

        // Dynamically create level buttons
        const levelSelector = document.getElementById('level-selector');
        AVAILABLE_LEVELS.forEach(level => {
            const button = document.createElement('button');
            button.textContent = `Level ${level}`;
            button.setAttribute('data-level', level); // Store the level for JavaScript
            button.classList.add('level-button');
            levelSelector.appendChild(button);
        });

        // User Menu Toggle Functionality
        const userMenuToggle = document.getElementById('userMenuToggle');
        const userMenuDropdown = document.getElementById('userMenuDropdown');

        userMenuToggle.addEventListener('click', function(event) {
            userMenuDropdown.classList.toggle('show');
            event.stopPropagation(); // Prevent click from propagating to document
        });

        // Close the dropdown if the user clicks outside of it
        document.addEventListener('click', function(event) {
            if (!userMenuToggle.contains(event.target) && !userMenuDropdown.contains(event.target)) {
                userMenuDropdown.classList.remove('show');
            }
        });


        // Placeholder for script.js functionality - these would typically be in script.js
        // Example for activating level buttons and loading units/floor plans:

        // Event listener for level selection
        levelSelector.addEventListener('click', function(event) {
            if (event.target.classList.contains('level-button')) {
                // Remove 'active' class from all level buttons
                document.querySelectorAll('.level-button').forEach(btn => {
                    btn.classList.remove('active');
                });

                // Add 'active' class to the clicked button
                event.target.classList.add('active');

                const selectedLevel = event.target.getAttribute('data-level');
                console.log(`Level ${selectedLevel} selected for Mall ID: ${SELECTED_MALL_ID}`);

                // --- Load Floor Plan ---
                const floorPlanArea = document.getElementById('floor-plan-area');
                // Use a placeholder image, in a real app, this would be an actual floor plan URL
                const floorPlanSrc = `https://placehold.co/600x400/CCCCCC/FFFFFF?text=Level+${selectedLevel}+Floor+Plan`;
                floorPlanArea.innerHTML = `<img src="${floorPlanSrc}" alt="Floor Plan Level ${selectedLevel}" class="floor-plan-thumb">`;

                // Add event listener for the new image (thumbnail)
                floorPlanArea.querySelector('.floor-plan-thumb').addEventListener('click', function() {
                    const magnifiedImg = document.getElementById('magnified-img');
                    const magnifyModal = document.getElementById('magnify-modal');

                    magnifiedImg.src = this.src; // Copy the thumbnail's src to the magnified image
                    magnifyModal.classList.add('active'); // Show the modal
                });


                // --- Placeholder: Load Units for the selected level ---
                const unitButtonContainer = document.getElementById('unit-button-container');
                unitButtonContainer.innerHTML = ''; // Clear previous units

                // Simulate fetching units (in a real app, this would be an AJAX call)
                const dummyUnits = [
                    { id: 'U' + selectedLevel + '-01', name: 'Unit ' + selectedLevel + '-01', status: 'Available', size: '100 sqm', price: '$5000/month', contact: 'leasing@example.com' },
                    { id: 'U' + selectedLevel + '-02', name: 'Unit ' + selectedLevel + '-02', status: 'Leased', size: '120 sqm', price: 'N/A', contact: 'N/A' },
                    { id: 'U' + selectedLevel + '-03', name: 'Unit ' + selectedLevel + '-03', status: 'Available', size: '80 sqm', price: '$4000/month', contact: 'leasing@example.com' },
                ];

                dummyUnits.forEach(unit => {
                    const unitButton = document.createElement('button');
                    unitButton.classList.add('unit-button');
                    unitButton.textContent = unit.name;
                    unitButton.setAttribute('data-unit-id', unit.id);
                    unitButton.setAttribute('data-unit-status', unit.status);
                    unitButton.setAttribute('data-unit-size', unit.size);
                    unitButton.setAttribute('data-unit-price', unit.price);
                    unitButton.setAttribute('data-unit-contact', unit.contact);

                    // Add class based on status for visual distinction
                    if (unit.status === 'Leased') {
                        unitButton.style.backgroundColor = '#f8d7da'; /* Light red */
                        unitButton.style.borderColor = '#dc3545'; /* Red border */
                        unitButton.style.color = '#721c24'; /* Dark red text */
                    }

                    unitButtonContainer.appendChild(unitButton);
                });

                // --- Placeholder: Clear Unit Details Panel ---
                document.getElementById('details-content').innerHTML = '<p class="initial-message">Select a unit to see its details.</p><div id="download-button-container"></div>';
            }
        });

        // Event listener for unit selection
        document.getElementById('unit-button-container').addEventListener('click', function(event) {
            if (event.target.classList.contains('unit-button')) {
                // Remove 'active' class from all unit buttons
                document.querySelectorAll('.unit-button').forEach(btn => {
                    btn.classList.remove('active');
                });

                // Add 'active' class to the clicked button
                event.target.classList.add('active');

                const unitId = event.target.getAttribute('data-unit-id');
                const unitName = event.target.textContent; // Get the full unit name from button
                const unitStatus = event.target.getAttribute('data-unit-status');
                const unitSize = event.target.getAttribute('data-unit-size');
                const unitPrice = event.target.getAttribute('data-unit-price');
                const unitContact = event.target.getAttribute('data-unit-contact');

                const detailsContent = document.getElementById('details-content');
                detailsContent.innerHTML = `
                    <h3>${unitName}</h3>
                    <p><strong>Status:</strong> <span>${unitStatus}</span></p>
                    <p><strong>Size:</strong> <span>${unitSize}</span></p>
                    <p><strong>Price:</strong> <span>${unitPrice}</span></p>
                    <p><strong>Contact:</strong> <span>${unitContact}</span></p>
                    <div id="download-button-container">
                        ${unitStatus === 'Available' ? '<button>Download Unit Brochure</button>' : ''}
                    </div>
                `;

                // Add event listener for download button (if present)
                const downloadButton = detailsContent.querySelector('#download-button-container button');
                if (downloadButton) {
                    downloadButton.addEventListener('click', () => {
                        alert('Downloading brochure for ' + unitName); // Replace with actual download logic
                    });
                }
            }
        });

        // Modal close functionality
        const magnifyModal = document.getElementById('magnify-modal');
        document.querySelector('.modal-close').addEventListener('click', function() {
            magnifyModal.classList.remove('active'); // Hide the modal
        });

        // Close modal when clicking outside the image (on the backdrop)
        magnifyModal.addEventListener('click', function(event) {
            if (event.target === this) {
                this.classList.remove('active'); // Hide the modal
            }
        });

        // The floor plan and units will now only load when a level button is explicitly clicked by the user.
        // No automatic click on first level button on page load.
    </script>
</body>
</html>